/**
 * Startup - SIMPLIFIED configuration page
 * HTML is already rendered - just wire up buttons and load saved values
 */

import { Page } from './Page.js';
import { showToast } from '../logging.js';

export class Startup extends Page {

  constructor(state, leedzConfig = null) {
    super('startup', state, leedzConfig);
    this.awsApiGatewayUrl = (leedzConfig || this.leedzConfig)?.aws?.apiGatewayUrl || null;
  }

  /**
   * Initialize - wire up button clicks
   */
  async initialize() {
    document.getElementById('startupClearBtn')?.addEventListener('click', () => this.clear());
    document.getElementById('startupSaveBtn')?.addEventListener('click', () => this.save());
  }

  /**
   * onShowImpl - load saved config, check server status
   * Runs AFTER page is visible
   */
  onShowImpl() {
    // Load saved values from state (if any)
    this.loadSavedConfig();

    // Check server status in background (fire and forget)
    this.checkServerStatus();

    // Fetch JWT token in background (fire and forget)
    this.fetchJWTToken();
  }

  /**
   * Load saved config from state into form
   */
  loadSavedConfig() {
    const config = this.state.Config || {};

    // Only update fields that have saved values
    if (config.serverHost) document.getElementById('startup-serverHost').value = config.serverHost;
    if (config.serverPort) document.getElementById('startup-serverPort').value = config.serverPort;
    if (config.mcpHost) document.getElementById('startup-mcpHost').value = config.mcpHost;
    if (config.mcpPort) document.getElementById('startup-mcpPort').value = config.mcpPort;
    if (config.llmApiKey) document.getElementById('startup-llmApiKey').value = config.llmApiKey;
    if (config.llmProvider) document.getElementById('startup-llmProvider').value = config.llmProvider;
    if (config.llmBaseUrl) document.getElementById('startup-llmBaseUrl').value = config.llmBaseUrl;
    if (config.llmAnthropicVersion) document.getElementById('startup-llmAnthropicVersion').value = config.llmAnthropicVersion;
    if (config.llmMaxTokens) document.getElementById('startup-llmMaxTokens').value = config.llmMaxTokens;
  }

  /**
   * Check server status in background
   */
  async checkServerStatus() {
    const host = document.getElementById('startup-serverHost').value || '127.0.0.1';
    const port = document.getElementById('startup-serverPort').value || '3000';
    const dbNameEl = document.getElementById('startup-dbName');

    if (!dbNameEl) return;

    try {
      const controller = new AbortController();
      setTimeout(() => controller.abort(), 2000);

      const response = await fetch(`http://${host}:${port}/config`, { signal: controller.signal });
      const config = await response.json();

      dbNameEl.textContent = config.databaseName || 'Unknown';
      dbNameEl.style.color = 'green';
    } catch (error) {
      dbNameEl.textContent = error.name === 'AbortError' ? 'Server offline' : 'Connection failed';
      dbNameEl.style.color = 'orange';
    }
  }

  /**
   * Fetch JWT token in background
   */
  async fetchJWTToken() {
    try {
      // Check if token already valid
      const stored = await chrome.storage.local.get(['leedzJWT', 'leedzJWTExpiry']);
      const now = Date.now();
      const sevenDays = 7 * 24 * 60 * 60 * 1000;

      if (stored.leedzJWT && stored.leedzJWTExpiry > (now + sevenDays)) {
        console.log('JWT token valid');
        return;
      }

      // Get user email
      const userInfo = await new Promise((resolve, reject) => {
        chrome.identity.getProfileUserInfo({ accountStatus: 'ANY' }, (info) => {
          chrome.runtime.lastError ? reject(chrome.runtime.lastError) : resolve(info);
        });
      });

      if (!userInfo.email || !this.awsApiGatewayUrl) return;

      // Fetch new token
      const controller = new AbortController();
      setTimeout(() => controller.abort(), 5000);

      const response = await fetch(`${this.awsApiGatewayUrl}/getToken?email=${encodeURIComponent(userInfo.email)}`, {
        signal: controller.signal
      });

      const { token, expires } = await response.json();

      await chrome.storage.local.set({
        leedzJWT: token,
        leedzJWTExpiry: expires * 1000,
        leedzUserEmail: userInfo.email
      });

      console.log('JWT token refreshed');
    } catch (error) {
      console.log('JWT token fetch failed (non-critical):', error.message);
    }
  }

  /**
   * Clear form to defaults
   */
  clear() {
    document.getElementById('startup-serverHost').value = '127.0.0.1';
    document.getElementById('startup-serverPort').value = '3000';
    document.getElementById('startup-mcpHost').value = '127.0.0.1';
    document.getElementById('startup-mcpPort').value = '3001';
    document.getElementById('startup-llmApiKey').value = '';
    document.getElementById('startup-llmProvider').value = 'claude-opus-4-1-20250805';
    document.getElementById('startup-llmBaseUrl').value = 'https://api.anthropic.com';
    document.getElementById('startup-llmAnthropicVersion').value = '2023-06-01';
    document.getElementById('startup-llmMaxTokens').value = '1024';
    document.getElementById('startup-dbName').textContent = 'Not connected';
    document.getElementById('startup-dbName').style.color = '#666';
  }

  /**
   * Save config
   */
  async save() {
    const config = {
      serverHost: document.getElementById('startup-serverHost').value.trim(),
      serverPort: document.getElementById('startup-serverPort').value.trim(),
      mcpHost: document.getElementById('startup-mcpHost').value.trim(),
      mcpPort: document.getElementById('startup-mcpPort').value.trim(),
      llmApiKey: document.getElementById('startup-llmApiKey').value.trim(),
      llmProvider: document.getElementById('startup-llmProvider').value.trim(),
      llmBaseUrl: document.getElementById('startup-llmBaseUrl').value.trim(),
      llmAnthropicVersion: document.getElementById('startup-llmAnthropicVersion').value.trim(),
      llmMaxTokens: parseInt(document.getElementById('startup-llmMaxTokens').value) || 1024,
      dbProvider: 'local_prisma_sqlite'
    };

    // Save to Chrome storage
    await chrome.storage.local.set({ leedzStartupConfig: config });

    // Merge with existing Config (preserve PDF settings)
    Object.assign(this.state.Config, config);

    // Save to database
    await this.state.save();

    // Reinitialize DB_LAYER
    const { getDbLayer } = await import('../provider_registry.js');
    window.DB_LAYER = await getDbLayer();

    // Check server status
    this.checkServerStatus();

    showToast('Configuration saved', 'success');
  }

  isStartupPage() { return true; }
  updateFromState(state) { this.state = state; this.loadSavedConfig(); }
  getActionButtons() { return null; }
}
