/**
 * Startup - Configuration page for server/database connection settings
 * Includes Gmail authorization functionality (previously in Gmailer page)
 */

import { Page } from './Page.js';
import { log, logError, showToast } from '../logging.js';
import { Gmailer } from './Gmailer.js';

export class Startup extends Page {

  constructor(state, leedzConfig = null) {
    super('startup', state, leedzConfig);

    // Create Gmailer instance to handle Gmail authorization
    this.gmailer = new Gmailer(state);

    // Store AWS config for JWT token endpoint (from parent's leedzConfig or constructor param)
    this.awsApiGatewayUrl = (leedzConfig || this.leedzConfig)?.aws?.apiGatewayUrl || null;
  }

  /**
   * Initialize startup page (called once on app startup)
   */
  async initialize() {
    this.setupButtons();

    // Initialize Gmailer controls
    await this.gmailer.initialize();
  }

  /**
   * Startup page never auto-parses
   */
  isStartupPage() {
    return true;
  }

  /**
   * Called when startup page becomes visible
   * SYNCHRONOUS - returns immediately to unblock page rendering
   */
  onShowImpl() {
    // FAST PATH: Populate form with defaults immediately (synchronous UI render)
    this.populateFormDefaults();

    // BACKGROUND TASKS: Run all async operations in parallel (non-blocking, fire-and-forget)
    // These update the UI progressively as they complete
    Promise.all([
      this.loadConfigFromState(),
      this.gmailer.loadMcpConfigAndCheckServer(),
      this.ensureJWTToken()
    ]).catch(error => {
      console.warn('Background startup tasks encountered errors:', error);
    });

    // Return immediately - don't block page rendering
  }

  /**
   * Update UI from state changes
   */
  updateFromState(state) {
    this.state = state;
    this.loadConfigFromState();
  }

  /**
   * Clear/reset startup config to defaults
   */
  clear() {
    // Reset to default values
    document.getElementById('startup-serverHost').value = '127.0.0.1';
    document.getElementById('startup-serverPort').value = '3000';
    document.getElementById('startup-mcpHost').value = '127.0.0.1';
    document.getElementById('startup-mcpPort').value = '3001';
    document.getElementById('startup-llmApiKey').value = '';
    document.getElementById('startup-llmProvider').value = 'claude-opus-4-1-20250805';
    document.getElementById('startup-llmBaseUrl').value = 'https://api.anthropic.com';
    document.getElementById('startup-llmAnthropicVersion').value = '2023-06-01';
    document.getElementById('startup-llmMaxTokens').value = '1024';
    document.getElementById('startup-dbName').textContent = 'Not connected';
    document.getElementById('startup-dbName').style.color = '#666';

    // Clear Gmail authorization state
    this.gmailer.clear();
  }

  /**
   * No action buttons for startup (uses custom buttons in button-wrapper)
   */
  getActionButtons() {
    return null;
  }

  /**
   * Wire up startup page button event handlers
   */
  setupButtons() {
    const clearBtn = document.getElementById('startupClearBtn');
    const saveBtn = document.getElementById('startupSaveBtn');

    if (clearBtn) {
      clearBtn.addEventListener('click', () => this.clear());
    }

    if (saveBtn) {
      saveBtn.addEventListener('click', () => this.saveConfig());
    }
  }

  /**
   * Populate form with default values immediately (synchronous, fast)
   * This ensures the UI renders instantly without waiting for DB/network calls
   */
  populateFormDefaults() {
    const serverHost = document.getElementById('startup-serverHost');
    const serverPort = document.getElementById('startup-serverPort');
    const mcpHost = document.getElementById('startup-mcpHost');
    const mcpPort = document.getElementById('startup-mcpPort');
    const llmApiKey = document.getElementById('startup-llmApiKey');
    const llmProvider = document.getElementById('startup-llmProvider');
    const llmBaseUrl = document.getElementById('startup-llmBaseUrl');
    const llmAnthropicVersion = document.getElementById('startup-llmAnthropicVersion');
    const llmMaxTokens = document.getElementById('startup-llmMaxTokens');
    const dbName = document.getElementById('startup-dbName');

    if (serverHost) serverHost.value = '127.0.0.1';
    if (serverPort) serverPort.value = '3000';
    if (mcpHost) mcpHost.value = '127.0.0.1';
    if (mcpPort) mcpPort.value = '3001';
    if (llmApiKey) llmApiKey.value = '';
    if (llmProvider) llmProvider.value = 'claude-opus-4-1-20250805';
    if (llmBaseUrl) llmBaseUrl.value = 'https://api.anthropic.com';
    if (llmAnthropicVersion) llmAnthropicVersion.value = '2023-06-01';
    if (llmMaxTokens) llmMaxTokens.value = '1024';
    if (dbName) {
      dbName.textContent = 'Checking...';
      dbName.style.color = '#666';
    }
  }

  /**
   * Load config from state and populate form fields
   * Called asynchronously in background - only updates UI if config found
   */
  async loadConfigFromState() {
    try {
      // State is already loaded by sidebar.js - just use it
      // DO NOT call state.load() again - it's redundant and wastes time

      // Get config values from state
      const config = this.state.Config;

      // If no config found, keep defaults already populated
      if (!config || Object.keys(config).length === 0) {
        console.log('No config in database, using defaults');
        return;
      }

      // Migrate old serverUrl format to serverHost (strip http:// or https://)
      let serverHost = config.serverHost;
      if (!serverHost && config.serverUrl) {
        serverHost = config.serverUrl.replace(/^https?:\/\//, '');
      }

      // Only update fields that have non-default values
      if (serverHost) {
        document.getElementById('startup-serverHost').value = serverHost;
      }
      if (config.serverPort) {
        document.getElementById('startup-serverPort').value = config.serverPort;
      }
      if (config.mcpHost) {
        document.getElementById('startup-mcpHost').value = config.mcpHost;
      }
      if (config.mcpPort) {
        document.getElementById('startup-mcpPort').value = config.mcpPort;
      }
      if (config.llmApiKey) {
        document.getElementById('startup-llmApiKey').value = config.llmApiKey;
      }
      if (config.llmProvider) {
        document.getElementById('startup-llmProvider').value = config.llmProvider;
      }
      if (config.llmBaseUrl) {
        document.getElementById('startup-llmBaseUrl').value = config.llmBaseUrl;
      }
      if (config.llmAnthropicVersion) {
        document.getElementById('startup-llmAnthropicVersion').value = config.llmAnthropicVersion;
      }
      if (config.llmMaxTokens) {
        document.getElementById('startup-llmMaxTokens').value = config.llmMaxTokens;
      }

      console.log('Loaded startup config from database');

      // Fetch database name from server (runs in parallel)
      const finalServerHost = serverHost || '127.0.0.1';
      const finalServerPort = config.serverPort || '3000';
      await this.fetchDatabaseName(finalServerHost, finalServerPort);

    } catch (error) {
      console.warn('Could not load config from database, keeping defaults:', error);
      // Don't clear - defaults already populated in populateFormDefaults()
    }
  }

  /**
   * Fetch database name from server with timeout
   */
  async fetchDatabaseName(serverHost, serverPort) {
    const dbNameElement = document.getElementById('startup-dbName');

    if (!dbNameElement) {
      console.warn('Database name element not found in DOM');
      return;
    }

    try {
      const serverUrl = `http://${serverHost}:${serverPort}`;

      // Abort fetch after 2 seconds if server not responding
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 2000);

      const response = await fetch(`${serverUrl}/config`, {
        signal: controller.signal
      });
      clearTimeout(timeoutId);

      if (!response.ok) {
        dbNameElement.textContent = 'No database configured';
        dbNameElement.style.color = 'red';
        return;
      }

      const config = await response.json();

      if (config && config.databaseName) {
        dbNameElement.textContent = config.databaseName;
        dbNameElement.style.color = 'green';
      } else {
        dbNameElement.textContent = 'Unknown';
        dbNameElement.style.color = 'orange';
      }

    } catch (error) {
      if (error.name === 'AbortError') {
        dbNameElement.textContent = 'Server not running';
        dbNameElement.style.color = 'orange';
        console.log('[Startup] Leedz server not running (timeout after 2s)');
        showToast('Leedz server not running', 'warning');
      } else {
        dbNameElement.textContent = 'Connection refused';
        dbNameElement.style.color = 'red';
        console.log('[Startup] Database connection failed:', error.message);
        showToast('Database connection failed', 'error');
      }
    }
  }

  /**
   * Save config to state, Chrome storage, and database
   */
  async saveConfig() {
    try {
      // Get values from form
      const serverHost = document.getElementById('startup-serverHost').value.trim();
      const serverPort = document.getElementById('startup-serverPort').value.trim();
      const mcpHost = document.getElementById('startup-mcpHost').value.trim();
      const mcpPort = document.getElementById('startup-mcpPort').value.trim();
      const llmApiKey = document.getElementById('startup-llmApiKey').value.trim();
      const llmProvider = document.getElementById('startup-llmProvider').value.trim();
      const llmBaseUrl = document.getElementById('startup-llmBaseUrl').value.trim();
      const llmAnthropicVersion = document.getElementById('startup-llmAnthropicVersion').value.trim();
      const llmMaxTokens = parseInt(document.getElementById('startup-llmMaxTokens').value.trim()) || 1024;

      // Validate required fields
      if (!serverHost || !serverPort) {
        showToast('Server Host and Port are required', 'error');
        return;
      }

      // CRITICAL: Load existing Config from database first to avoid overwriting PDF settings
      await this.state.loadConfigFromDB();

      // Update ONLY the Startup-related fields (preserve existing PDF settings)
      if (!this.state.Config) {
        this.state.Config = {};
      }

      this.state.Config.serverHost = serverHost;
      this.state.Config.serverPort = serverPort;
      this.state.Config.dbProvider = 'local_prisma_sqlite';
      this.state.Config.mcpHost = mcpHost || '127.0.0.1';
      this.state.Config.mcpPort = mcpPort || '3001';
      this.state.Config.llmApiKey = llmApiKey || null;
      this.state.Config.llmProvider = llmProvider || 'claude-opus-4-1-20250805';
      this.state.Config.llmBaseUrl = llmBaseUrl || 'https://api.anthropic.com';
      this.state.Config.llmAnthropicVersion = llmAnthropicVersion || '2023-06-01';
      this.state.Config.llmMaxTokens = llmMaxTokens;

      // Save to Chrome storage (only startup settings)
      await chrome.storage.local.set({
        leedzStartupConfig: {
          serverHost, serverPort, dbProvider: 'local_prisma_sqlite', mcpHost, mcpPort,
          llmApiKey, llmProvider, llmBaseUrl, llmAnthropicVersion, llmMaxTokens
        }
      });

      // Save to database (merged Config with both Startup + PDF settings)
      await this.state.save();

      // CRITICAL: Reinitialize DB_LAYER to point to new database
      const { getDbLayer } = await import('../provider_registry.js');
      window.DB_LAYER = await getDbLayer();
      console.log('DB_LAYER reinitialized to:', window.DB_LAYER.baseUrl);

      // CRITICAL: Clear Config cache so Invoicer will reload from NEW database
      this.state.Config = {};
      console.log('Config cache cleared - will reload from new database on next use');

      // Fetch and display database name
      await this.fetchDatabaseName(serverHost, serverPort);

      console.log('Startup config saved');
      showToast('Configuration saved successfully', 'success');

    } catch (error) {
      console.error('Failed to save startup config:', error);
      showToast('Failed to save configuration', 'error');
      logError('Config save failed:', error);
    }
  }

  /**
   * Ensure valid JWT token exists for LEEDZ backend authentication
   * Fetches new token if missing or expiring soon (within 7 days)
   * @returns {Promise<string|null>} - JWT token or null if unavailable
   */
  async ensureJWTToken() {
    try {
      // Check if token exists and is valid
      const stored = await chrome.storage.local.get(['leedzJWT', 'leedzJWTExpiry', 'leedzUserEmail']);

      const now = Date.now();
      const sevenDaysMs = 7 * 24 * 60 * 60 * 1000;

      if (stored.leedzJWT && stored.leedzJWTExpiry && stored.leedzJWTExpiry > (now + sevenDaysMs)) {
        // Token valid for at least 7 more days
        console.log('JWT token valid until:', new Date(stored.leedzJWTExpiry));
        return stored.leedzJWT;
      }

      // Get user email from Chrome identity
      const userInfo = await new Promise((resolve, reject) => {
        chrome.identity.getProfileUserInfo({ accountStatus: 'ANY' }, (info) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(info);
          }
        });
      });

      if (!userInfo.email) {
        throw new Error('Could not get user email from Chrome identity');
      }

      // Call getToken API on LEEDZ AWS backend (NOT local server)
      if (!this.awsApiGatewayUrl) {
        throw new Error('AWS API Gateway URL not configured in leedz_config.json');
      }

      const apiUrl = `${this.awsApiGatewayUrl}/getToken?email=${encodeURIComponent(userInfo.email)}`;

      console.log('Fetching JWT token from AWS API Gateway:', apiUrl);

      // Abort fetch after 5 seconds if AWS not responding
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 5000);

      const response = await fetch(apiUrl, {
        signal: controller.signal
      });
      clearTimeout(timeoutId);

      if (!response.ok) {
        throw new Error(`AWS getToken API returned ${response.status}`);
      }

      const { token, expires } = await response.json();

      // Store token in Chrome storage
      await chrome.storage.local.set({
        leedzJWT: token,
        leedzJWTExpiry: expires * 1000, // Convert to milliseconds
        leedzUserEmail: userInfo.email
      });

      console.log('JWT token obtained and stored, expires:', new Date(expires * 1000));
      return token;

    } catch (error) {
      console.error('Failed to ensure JWT token:', error);
      // Don't throw - allow extension to continue without LEEDZ marketplace access
      return null;
    }
  }
}
