/**
 * ClientCapture - Page class for capturing multiple clients
 * Each client gets its own table frame
 * Extends DataPage for universal workflow
 */

import { DataPage } from './DataPage.js';
import { ValidationUtils } from '../utils/ValidationUtils.js';
import { PageUtils } from '../utils/Page_Utils.js';
import { log, logError, showToast } from '../logging.js';
import Client from '../db/Client.js';

export class ClientCapture extends DataPage {

  constructor(state) {
    super('clients', state);
    this.clients = []; // Array of client objects (one per table/frame)
  }

  /**
   * Initialize client capture page (called once on app startup)
   */
  async initialize() {
    // Wire up the Plus button in header
    const addBtn = document.getElementById('addClientBtnHeader');
    if (addBtn) {
      addBtn.addEventListener('click', () => this.addClientFrame());
    }
  }

  /**
   * Called when page is hidden - save form data to state
   */
  async onHide() {
    // Sync current form data to state before leaving page
    this.syncFormToState();
  }

  /**
   * DataPage hook: Run full parse (LLM extraction)
   * @returns {Promise<Object>} { success: boolean, data: Object, error: string }
   */
  async fullParse() {
    console.log('=== ClientCapture.fullParse() called ===');
    const result = await this.reloadParser();
    console.log('=== reloadParser result:', result);

    if (result.success) {
      // Include Client (singular) so DataPage.searchDB can find name/email
      const firstClient = result.clients[0] || {};
      return {
        success: true,
        data: {
          Clients: result.clients,
          Client: { name: firstClient.name, email: firstClient.email }
        },
        fromDB: result.fromDB
      };
    }
    return result;
  }

  /**
   * DataPage hook: Render data from STATE cache
   * @param {Object} stateData - State data to render
   */
  async renderFromState(stateData) {
    // Load from state (or show blank if no data)
    if (stateData?.Clients?.length > 0) {
      this.clients = stateData.Clients.map(c => ({ ...c }));
    } else if (stateData?.Client?.name || stateData?.Client?.email) {
      this.clients = [{ ...stateData.Client }];
    } else {
      // No meaningful state data - show blank frame
      this.clients = [this._createBlankClient()];
    }

    this.render();
  }

  /**
   * DataPage hook: Render data from database (with green styling)
   * @param {Object} dbData - Database client data
   */
  async renderFromDB(dbData) {
    // Mark as from DB for green styling
    const client = {
      name: dbData.name || '',
      email: dbData.email || '',
      phone: dbData.phone || '',
      company: dbData.company || '',
      website: dbData.website || '',
      clientNotes: dbData.clientNotes || '',
      _fromDB: true
    };

    this.clients = [client];

    // Update state
    Object.assign(this.state.Client, client);
    this.state.setClients([client]);

    this.render();
  }

  /**
   * DataPage hook: Render data from fresh parse
   * @param {Object} parseResult - Parse result data
   */
  async renderFromParse(parseResult) {
    if (parseResult?.success && parseResult?.data?.Clients?.length > 0) {
      this.clients = parseResult.data.Clients.map(c => ({ ...c }));
    } else {
      // No data or parse failed - show blank frame
      this.clients = [this._createBlankClient()];
    }

    this.render();
  }


  /**
   * Update UI from state changes
   */
  updateFromState(state) {
    this.state = state;
    // Re-render all frames
    this.render();
  }

  /**
   * Clear/reset to one blank frame
   */
  clear() {
    this.clients = [this._createBlankClient()];
    this.render();
    log('Cleared');
  }

  /**
   * Get action buttons for client capture page
   */
  getActionButtons() {
    return [
      { id: 'clearClientsBtn', label: 'Clear', handler: () => this.clear() },
      { id: 'saveClientsBtn', label: 'Save', handler: () => this.saveAllClients() }
    ];
  }

  /**
   * Create a blank client object
   */
  _createBlankClient() {
    return {
      name: '',
      email: '',
      phone: '',
      company: '',
      clientNotes: '',
      _fromDB: false  // Explicitly mark as not from DB
    };
  }

  /**
   * Sync form data from this.clients to state.Clients
   */
  syncFormToState() {
    // Copy clients array to state, filtering out any nulls
    this.state.setClients(this.clients.filter(c => c !== null));

    // Also update state.Client (first client) for backward compatibility
    if (this.clients.length > 0 && this.clients[0]) {
      Object.assign(this.state.Client, this.clients[0]);
    }
  }

  /**
   * Render all client tables (frames)
   */
  render() {
    const container = document.getElementById('display_win_clients');
    if (!container) return;

    // Filter out null clients (deleted entries)
    this.clients = this.clients.filter(c => c !== null);

    // Ensure at least one blank frame
    if (this.clients.length === 0) {
      this.clients.push(this._createBlankClient());
    }

    // Clear container (except loading spinner)
    const spinner = container.querySelector('.loading-spinner');
    container.innerHTML = '';

    // Re-add spinner
    if (spinner) container.appendChild(spinner);

    // Render each client as a separate table
    this.clients.forEach((client, index) => {
      const table = this._renderClientTable(client, index);
      container.appendChild(table);
    });
  }

  /**
   * Render a single client table
   */
  _renderClientTable(client, index) {
    // Create container for frame (table + delete button)
    const frameContainer = document.createElement('div');
    frameContainer.className = 'client-frame';
    frameContainer.setAttribute('data-client-index', index);

    const table = document.createElement('table');
    table.className = 'booking-table';
    table.setAttribute('data-client-index', index);

    // Apply visual indicators if client is from DB
    if (client._fromDB) {
      table.style.backgroundColor = 'honeydew';
      table.style.borderColor = 'var(--LEEDZ_GREEN)';
    }

    // Header
    const thead = document.createElement('thead');
    thead.innerHTML = `
      <tr>
        <th>Field</th>
        <th>Value</th>
      </tr>
    `;
    table.appendChild(thead);

    // Body with all client fields
    const tbody = document.createElement('tbody');
    const clientFields = Client.getFieldNames();

    clientFields.forEach(field => {
      const row = document.createElement('tr');

      // Field name cell
      const nameCell = document.createElement('td');
      nameCell.className = 'field-name';
      nameCell.textContent = field;

      // Field value cell
      const valueCell = document.createElement('td');
      valueCell.className = 'field-value';

      // Use textarea for clientNotes, input for others
      let inputElement;
      if (field === 'clientNotes') {
        inputElement = document.createElement('textarea');
        inputElement.rows = 3;
      } else {
        inputElement = document.createElement('input');
        inputElement.type = 'text';
      }

      inputElement.setAttribute('data-field', field);
      inputElement.setAttribute('data-client-index', index);
      inputElement.value = client[field] || '';

      // Format phone for display
      if (field === 'phone' && client[field]) {
        inputElement.value = ValidationUtils.formatPhoneForDisplay(client[field]);
      }

      // Add input listener to sync to clients array
      inputElement.addEventListener('input', (e) => {
        const idx = parseInt(e.target.dataset.clientIndex);
        const fieldName = e.target.dataset.field;
        let value = e.target.value;

        // Remove phone formatting for storage
        if (fieldName === 'phone') {
          value = value.replace(/[^\d]/g, '');
        }

        this.clients[idx][fieldName] = value;
      });

      // Add blur listener for formatting
      inputElement.addEventListener('blur', (e) => {
        const fieldName = e.target.dataset.field;
        if (fieldName === 'phone' && e.target.value) {
          e.target.value = ValidationUtils.formatPhoneForDisplay(e.target.value);
        }
      });

      valueCell.appendChild(inputElement);
      row.appendChild(nameCell);
      row.appendChild(valueCell);
      tbody.appendChild(row);
    });

    table.appendChild(tbody);

    // Add table to frame container
    frameContainer.appendChild(table);

    // Create button container for Delete + Email buttons (horizontal row)
    const buttonRow = document.createElement('div');
    buttonRow.className = 'client-button-row';

    // Create delete button
    const deleteBtn = document.createElement('button');
    deleteBtn.id = `deleteClientBtn-${index}`;
    deleteBtn.className = 'sidebar-button delete-client-btn';
    deleteBtn.textContent = 'Delete';
    deleteBtn.setAttribute('data-client-index', index);

    // Wire delete button click handler
    deleteBtn.addEventListener('click', () => this.deleteClient(index));

    // Create email button (takes user to Outreach page with this client)
    const emailBtn = document.createElement('button');
    emailBtn.id = `emailClientBtn-${index}`;
    emailBtn.className = 'sidebar-button email-client-btn';
    emailBtn.textContent = 'Email';
    emailBtn.setAttribute('data-client-index', index);

    // Wire email button click handler
    emailBtn.addEventListener('click', () => this.emailClient(index));

    // Add buttons to button row
    buttonRow.appendChild(deleteBtn);
    buttonRow.appendChild(emailBtn);

    // Add button row to frame container
    frameContainer.appendChild(buttonRow);

    return frameContainer;
  }

  /**
   * Add a new blank client frame at the top
   */
  addClientFrame() {
    this.clients.unshift(this._createBlankClient());
    this.render();
    log('Added new client frame');
  }

  /**
   * Delete a client frame
   * @param {number} index - Index of client to delete
   */
  deleteClient(index) {
    // Mark client as deleted (null)
    this.clients[index] = null;

    // Re-render (will filter out nulls and ensure at least one frame)
    this.render();

    // Show feedback
    showToast('Client deleted', 'info');
    log(`Client at index ${index} deleted`);
  }

  /**
   * Email a client - navigate to Outreach page with this client
   * @param {number} index - Index of client to email
   */
  async emailClient(index) {
    // Get the client to email
    const client = this.clients[index];

    if (!client) {
      showToast('Client not found', 'error');
      return;
    }

    // Sync current form data to state first
    this.syncFormToState();

    // Set the clicked client as the current client in state
    Object.assign(this.state.Client, client);

    // Send ALL clients to Outreach, but reorder so clicked client is first
    // This way Outreach will start with the clicked client at index 0
    const reorderedClients = [];

    // First element: the clicked client
    reorderedClients.push(this.clients[index]);

    // Then add all other clients
    for (let i = 0; i < this.clients.length; i++) {
      if (i !== index) {
        reorderedClients.push(this.clients[i]);
      }
    }

    this.state.setClients(reorderedClients);

    console.log(`Sending ${reorderedClients.length} clients to Outreach (clicked client at index 0)`);

    // Navigate to Responder page
    if (window.switchToPage) {
      await window.switchToPage('responder');
      log(`Navigated to Responder with ${reorderedClients.length} client(s)`);
    } else {
      showToast('Navigation error - please try again', 'error');
      console.error('window.switchToPage not available');
    }
  }

  /**
   * Save all clients to database via state.save()
   * State automatically loops through Clients array
   */
  async saveAllClients() {
    // Sync current form data to state first
    this.syncFormToState();

    // Use shared utility for save logic
    await PageUtils.saveClientData(this.state, {
      multiClient: true,
      showToast,
      log: (msg) => log(msg)
    });
    // Do NOT clear form after save - user may want to edit and re-save
  }

  /**
   * Enable or disable action buttons
   */
  setButtonsEnabled(enabled) {
    const clearBtn = document.getElementById('clearClientsBtn');
    const saveBtn = document.getElementById('saveClientsBtn');
    const deleteButtons = document.querySelectorAll('.delete-client-btn');

    if (clearBtn) clearBtn.disabled = !enabled;
    if (saveBtn) saveBtn.disabled = !enabled;
    deleteButtons.forEach(btn => btn.disabled = !enabled);
  }

  /**
   * Reload parser to extract clients from current page
   * Pure logic - no UI updates (caller handles spinner, render, toast)
   * @returns {Promise<Object>} { success: boolean, clients: Array, fromDB: boolean, error: string }
   */
  async reloadParser() {
    console.log('=== ClientCapture.reloadParser() called ===');
    try {
      // Get current tab URL and tabId
      const { url, tabId } = await new Promise(resolve => {
        chrome.runtime.sendMessage({ type: 'leedz_get_tab_url' }, resolve);
      });

      if (!url || !tabId) {
        return { success: false, error: 'No page detected' };
      }

      // Send message to content script to run matching parser (extractClientData only)
      const response = await new Promise((resolve) => {
        chrome.tabs.sendMessage(tabId, {
          type: 'leedz_extract_client',
          state: this.state.toObject()
        }, resolve);
      });

      if (response?.ok && response?.data) {
        // Extract clients array from response
        const clientsArray = response.data.Clients;

        if (clientsArray && Array.isArray(clientsArray) && clientsArray.length > 0) {
          // Check if client is from DB
          const fromDB = clientsArray[0]._fromDB === true;

          // Update internal clients array
          this.clients = clientsArray.map(client => ({
            name: client.name || '',
            email: client.email || '',
            phone: client.phone || '',
            company: client.company || '',
            website: client.website || '',
            clientNotes: client.clientNotes || '',
            _fromDB: client._fromDB || false
          }));

          return { success: true, clients: this.clients, fromDB };
        } else {
          return { success: false, error: 'No client data found on page' };
        }
      } else {
        return { success: false, error: response?.error || 'Parser failed' };
      }

    } catch (error) {
      console.error('Parser initialization error:', error);
      return { success: false, error: error.message };
    }
  }
}
